/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

#include "mx_auto_config.h"
#include "myriexpress.h"
#if MX_OS_WINNT
#include <windows.h>
#include "getopt.h"
#else
#include <unistd.h>
#endif
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include "mx_version.h"
#include "mx__driver_interface.h"
#include "mcp_config.h"

const char *mx__nic_id_to_str(char *str, uint64_t nic_id, uint32_t len);

int namechars = 25;
int shortnames = 0;

static int 
report_board(mx_endpt_handle_t fd, uint32_t board_number, 
	     int verbose, int print_peers, uint32_t max_peers, int all);
static void
report_peer_table(mx_endpt_handle_t fd, uint32_t board_number, uint64_t nic_id,
		  int num_ports, int verbose, uint32_t max_peers, int all);

#define ROUTE_MAX_LEN MX_MCP_ROUTE_MAX_LENGTH

#define MAXRTLEN 256
#define MAXRTNUM MAXRTLEN/8

#define PEER_ACTIVE_MASK(bnum) ( (MX_PEER_FLAG_SEEN_P0 | MX_PEER_FLAG_SEEN_P1) << ((bnum) * 2) )

struct route_info {
  char route[MAXRTLEN];
  int len;
};

struct route_info saved_routes[MAXRTNUM];


static void
report_error(int board, const char *s)
{
  fprintf(stderr, "Error: Board %d:%s\n", board, s);
  exit(1);
}

static int
printroutes(char *r, int routesize, int do_print, int *hop_cnt)
{
  int len;
  int i, count = 0;
  char *p;
  char *end = r + routesize;

  *hop_cnt = 0;
  memset(saved_routes, 0, sizeof(saved_routes));
  
  /* routes are packed starting on 8-byte boundaries.
     1 byte of length is found just before the next 8-byte boundary */

  len = 0xf & r[ROUTE_MAX_LEN];
  while ((r < end) && len) {
    if (len > *hop_cnt)
      *hop_cnt = len;
      
    for (p = r; p < r + len; p++) {
      if (do_print)
	printf("%x ", *p & 0xff);
    }
    for (i = 0; i < count; i++) {
      if (len != saved_routes[i].len)
	continue;
      if (0 == memcmp(saved_routes[i].route, r, len))
	break;
    }
    if (i == count) {
      count++;
      memcpy(saved_routes[i].route, r, len);
      saved_routes[i].len = len;
    }
    r = r + ROUTE_MAX_LEN + 1;
    if (r < end) {
      len = r[ROUTE_MAX_LEN];
      if (len && do_print)
	printf("** ");
    }
  }
  if (!count && (uint8_t)r[0] == 0xff) {
    if (do_print)
      printf(" PT2PT");
    count = 1;
  }
  return count;
}


void printname(uint64_t mac, int i)
{
  char buf[18];
  char name[MX_MAX_HOSTNAME_LEN];
  int namelen;
  mx_return_t rc;
  int x;
  
  printf("%4d) %s ", i, mx__nic_id_to_str(buf, mac, sizeof buf));
  
  rc = mx_nic_id_to_hostname(mac, name);
  if (rc != MX_SUCCESS) {
    strcpy(name, "<unknown>");
  }
  if (shortnames) {
    char *dcol = strrchr(name,':');
    char *pcol = strchr(name,'.');
    if (pcol) {
      char suffix[MX_MAX_HOSTNAME_LEN];
      strcpy(suffix,".<dom>");
      if (dcol && dcol > pcol) {
	strcat(suffix, dcol);
      }
      strcpy(pcol, suffix);
    }
  }
  namelen = strlen(name);
  printf(name);
  for(x = 0; x < namechars - namelen || x == 0; x++)
    printf(" ");
}

static void 
parse_route_table_flat(char *pptr, char *rptr0, int max_peers,
		       int routesize, mx_get_peer_format_t peer_format,
		       int num_ports, int board_num, int all)
{
  uint64_t mac;
  char *rptr1;
  int i, x, route_count0, route_count1, hop_cnt0, hop_cnt1;

  if (num_ports == 2)
    rptr1 = rptr0 + ((1 + max_peers) * routesize);
  else
    rptr1 = NULL;

  for (x = 0; x < strlen("INDEX    MAC ADDRESS    ") + namechars; x++)
    printf(" ");
  printf("ROUTE COUNT\n");

  printf("INDEX    MAC ADDRESS     HOST NAME");
  for (x = strlen("HOST NAME"); x < namechars; x++)
    printf(" ");

  printf(" P0");
  if (num_ports == 2)
    printf("   P1");
  printf("\n");
  printf("-----    -----------     ---------");
  for (x = strlen("HOST NAME"); x < namechars; x++)
    printf(" ");
  printf(" ---");
  if (num_ports == 2)
    printf("  ---");
  printf("\n");
  route_count1 = 0;
  for (i = 0; i < max_peers; i++) {
    mx_peer_t * peer = (mx_peer_t *)pptr;
    route_count0 = printroutes(rptr0, routesize, 0, &hop_cnt0);
    if (rptr1 != NULL)
      route_count1 = printroutes(rptr1, routesize, 0, &hop_cnt1);
    mac = peer->mac_low32;
    mac += (uint64_t)peer->mac_high16 << 32;
    if (mac && (all || (peer->flags & PEER_ACTIVE_MASK(board_num)))) {
        printname(mac, i);
	printf("%c", (peer->flags & PEER_ACTIVE_MASK(board_num)) ? ' ' : 'D');
	printf(" %d,%d", route_count0, hop_cnt0);
	if (num_ports == 2)
	  printf("  %d,%d", route_count1, hop_cnt1);
	printf("\n");
      }
    pptr += peer_format.sizeof_peer_t;
    rptr0 += routesize;
    if (rptr1 != NULL)
      rptr1 += routesize;
  }
}


static void 
parse_route_table(char *pptr, char *rptr, int max_peers,
		  int routesize, mx_get_peer_format_t peer_format, int board_number, int all)
{
  uint64_t mac;
  int i, x, hop_cnt;
  
  printf("INDEX   MAC ADDRESS     HOST NAME");
  for (x = strlen("HOST NAME"); x < namechars; x++)
    printf(" ");
  printf(" ROUTES\n");
  printf("-----   -----------     ---------");
  for (x = strlen("HOST NAME"); x < namechars; x++)
    printf(" ");
  printf(" ------\n");
  for (i = 0; i < max_peers; i++) {
    mx_peer_t * peer = (mx_peer_t *)pptr;
    mac = peer->mac_low32;
    mac += (uint64_t)peer->mac_high16 << 32;
    if (mac && (all || (peer->flags & PEER_ACTIVE_MASK(board_number)))) {
      printname(mac, i);
      printf("%c ", (peer->flags & PEER_ACTIVE_MASK(board_number)) ? ' ' : 'D');
      (void)printroutes(rptr, routesize, 1, &hop_cnt);
      printf("\n");
    }
    pptr += peer_format.sizeof_peer_t;
    rptr += routesize;
  }
}

void 
usage()
{
  fprintf(stderr, "Usage: mx_info [args]\n");
  fprintf(stderr, "-b - Instance number [0]\n");
  fprintf(stderr, "-v - verbose, print detailed routes\n");
  fprintf(stderr, "-s - shortnames\n");
  fprintf(stderr, "-a - all peers (default)\n");
  fprintf(stderr, "-m - only \"mapper\" peers seen in last map update\n");
  fprintf(stderr, "-q - quiet, don't list peers\n");
  fprintf(stderr, "-w - increase hostname field width\n");
  fprintf(stderr, "-h - help\n");
}


uint32_t instance_count;

int 
main(int argc, char **argv)
{
  mx_endpt_handle_t fd;
  int board_number, found, look_for;
  uint32_t max_instance, max_peers, verbose, print_peers;
  mx_return_t ret;
  mx_get_version_t kversions;
  int c;

  int all = 1;
  verbose = 0;
  print_peers = 1;
  board_number = 0;
  look_for = 0;

  while ((c = getopt(argc, argv, "aqhsvwb:m")) != EOF) switch(c) {
  case 'b':
    board_number = atoi(optarg);
    look_for = 1;
    break;
  case 'v':
    verbose = 1;
    break;
  case 'a':
    all = 1;
    break;
  case 'm':
    all = 0;
    break;
  case 's':
    shortnames = 1;
    break;
  case 'q':
    print_peers = 0;
    break;
  case 'w':
    namechars *= 2;
    break;
  case 'h':
  default:
    usage();
    exit(1);
  }

  mx_init();
  mx_set_error_handler(MX_ERRORS_RETURN);
  ret = mx_open_board(board_number, &fd);
  if (ret != MX_SUCCESS) {
    printf("open failed: %s\n", mx_strerror(ret));
    exit(1);
  }

  ret = mx__get_version(fd, &kversions);
  if (ret != MX_SUCCESS) {
    printf("mx__get_version_info failed\n");
    exit(1);
  }
  printf("MX Version: %s\n", MX_VERSION_STR);
  printf("MX Build: %s\n", MX_BUILD_STR);
  if (strcmp(kversions.version_str, MX_VERSION_STR) != 0) {
    printf("MX Driver Version: %s\n", kversions.version_str);
  }
  if (strcmp(kversions.build_str, MX_BUILD_STR) != 0) {
    printf("MX Driver Build: %s\n", kversions.build_str);
  }
  
  ret = mx_get_info(NULL, MX_NIC_COUNT, NULL, 0, &instance_count, 
		    sizeof (instance_count));

  if (ret != MX_SUCCESS) {
    printf("mx_get_info(...MX_NIC_COUNT..) failed\n");
    exit(1);
  }

  if (!look_for)
    look_for = instance_count;

  if (mx__get_max_instance(fd, &max_instance) != 0) {
    printf("MX_GET_MAX_INSTANCE failed\n");
    exit(1);
  }

  if (mx__get_max_peers(fd, &max_peers) != 0) {
    printf("MX_GET_MAX_PEERS failed\n");
    exit(1);
  }

  printf("%d Myrinet board%s installed.\n", instance_count,
	 instance_count > 1 ? "s" : "");
  printf("The MX driver is configured to support up to %d instances and %d nodes.\n", 
	 max_instance, max_peers);
  for (found = 0;
       (board_number < max_instance) && (found < look_for); 
       board_number++) {
    printf("===================================================================\n");
    found += report_board(fd, board_number, verbose, print_peers, max_peers, all);
  }

  mx__close(fd);
  return(0);
}

static char *mx__dead_string(unsigned status)
{
  static char buf[100];
  switch (status) {
  case 0:
    return "Running";
  /* the two following are obsolete code kept for compat with old drivers */
  case 1: 
    return "Dead:Firmware not responding";
  case 2: 
    return "Parity error recovery in progress";
  /* real codes */
  case MX_DEAD_RECOVERABLE_SRAM_PARITY_ERROR: 
    return "Dead:Recoverable SRAM Parity Error";
  case MX_DEAD_SRAM_PARITY_ERROR:
    return "Dead:Fatal SRAM Parity Error (reboot needed)";
  case MX_DEAD_WATCHDOG_TIMEOUT:
    return "Dead:Watchdog Timeout";
  case MX_DEAD_COMMAND_TIMEOUT:
    return "Dead:Command Timeout";
  case MX_DEAD_ENDPOINT_CLOSE_TIMEOUT:
    return "Dead:Endpoint Close Timeout";
  case MX_DEAD_ROUTE_UPDATE_TIMEOUT:
    return "Dead:Route Update Timeout";
  case MX_DEAD_PCI_PARITY_ERROR:
    return "Dead:Pci Parity Error";
  case MX_DEAD_PCI_MASTER_ABORT:
    return "Dead:Pci Master Abort";
#ifdef MX_DEAD_NIC_RESET
  case MX_DEAD_NIC_RESET:
    return "Dead:NIC Resetted";
#endif
  default:
    sprintf(buf, "Dead: Unknown code %d", status);
    return buf;
  }
}

static int 
report_board(mx_endpt_handle_t fd, uint32_t board_number, 
	     int verbose, int print_peers, uint32_t max_peers, int all)
{
  mx_get_nic_id_t x;
  mx_mapper_state_t ms;
  mx_return_t ret;
  uint32_t num_ports, serial_number;
  uint32_t xfer;
  int i;
  int board_status = 0;
  float cpu_freq, pci_freq;
  char product_code[MX_MAX_STR_LEN];
  char part_number[MX_MAX_STR_LEN];
  mx_get_board_val_t sram;
  mx_get_board_val_t type;

  printf("Instance #%d: ", board_number);
  
  type.board_number = board_number;
  if (mx__get_board_type(fd, &type) != 0)
    report_error(board_number, "Cannot get board_type");

  xfer = board_number;
  if (mx__get_cpu_freq(fd, &xfer) != 0)
    report_error(board_number, "Cannot get cpu freq");
  cpu_freq = (float)xfer;
  cpu_freq = cpu_freq/1024.0;

  xfer = board_number;
  if (mx__get_pci_freq(fd, &xfer) != 0) {
    report_error(board_number, "Cannot get pci freq");
  }
  printf(" %3.1f MHz LANai,", cpu_freq);
  if (type.val == MX_BOARD_TYPE_Z) {
    printf(" PCI-E x%d", xfer);
  } else {
    pci_freq = ((float)xfer / 8192.0) * cpu_freq;
    printf(" %3.1f MHz PCI bus", pci_freq);
  }

  sram.board_number = board_number;

  if (!mx__get_sram_size(fd, &sram)) {
    printf(", %1.0f MB SRAM", sram.val / (1024.0 * 1024.0));
  }

  printf("\n\tStatus:\t\t");
  xfer = board_number;
  if (mx__get_board_status(fd, &xfer)) {
    printf("failed to retrieve status (%s),", strerror(errno));
    exit(1);
  } else {
    board_status = xfer;
    printf("%s", mx__dead_string(board_status));
  }

  num_ports = board_number;
  if (mx_get_info(NULL, MX_PORT_COUNT, &board_number, sizeof(board_number),
		  &num_ports, sizeof(num_ports)) != MX_SUCCESS) {
    printf("mx_get_info(MX_PORT_COUNT) failed\n");
    exit(1);
  }

  
  xfer = board_number;
  if (mx__get_link_state(fd, &xfer) == 0) {
    for (i = 0; i < num_ports; i++)
      if (xfer & (1 << i))
	printf(", P%d: Link up", i);
      else
	printf(", P%d: Link down", i);
  }

  x.board_number = board_number;
  if (mx__get_nic_id(fd, &x) != 0) {
    printf("not present\n");
    exit(1);
  }
  else {
    char buf[18];
    printf("\n\tMAC Address:\t%s\n", mx__nic_id_to_str(buf, x.nic_id, 18));
  }

  ret = mx_get_info(NULL, MX_PRODUCT_CODE, &board_number, sizeof(board_number),
		    product_code, sizeof (product_code));
  if (ret != MX_SUCCESS)
    sprintf(product_code, "unknown");
  printf("\tProduct code:\t%s\n", product_code);

  ret = mx_get_info(NULL, MX_PART_NUMBER, &board_number, sizeof(board_number),
		    part_number, sizeof (part_number));
  if (ret != MX_SUCCESS)
    sprintf(part_number, "unknown");
  printf("\tPart number:\t%s\n", part_number);

  ret = mx_get_info(NULL, MX_SERIAL_NUMBER, &board_number, sizeof(board_number),
		    &serial_number, sizeof (serial_number));
  if (ret != MX_SUCCESS) {
    printf("Unable to obtain serial number\n");
    exit(1);
  } else {
    printf("\tSerial number:\t%d\n", serial_number);
  }

  for (i = 0; i < num_ports; i++) {
    ms.board_number = board_number;
    ms.iport = i;
    if (mx__get_mapper_state(fd, &ms) != 0) {
      printf("MX_GET_MAPPER_STATE failed for board %d: port:%d\n",
	     board_number, i);
      exit(1);
    }
    if (num_ports == 1)
      printf("\tMapper:\t");
    else
      printf("\tMapper (P%d):", i);
    printf("\t%2.2x:%2.2x:%2.2x:%2.2x:%2.2x:%2.2x,",
	   ms.mapper_mac[0] & 0xff, ms.mapper_mac[1] & 0xff,
	   ms.mapper_mac[2] & 0xff, ms.mapper_mac[3] & 0xff,
	   ms.mapper_mac[4] & 0xff, ms.mapper_mac[5] & 0xff);
    printf(" version = 0x%8.8x,", ms.map_version);
    if (ms.network_configured)
      printf(" configured\n\tMapped hosts:\t%d\n", ms.num_hosts);
    else
      printf(" !configured\n");
  }
  printf("\n");
  if (print_peers)
    report_peer_table(fd, board_number, x.nic_id, num_ports, verbose, max_peers, all);

  if (board_status != 0) {
    printf("WARNING: Status = %s!\n", mx__dead_string(board_status));
    printf("WARNING: Instance %d is not running!\n", board_number);
  }	
  return 1;
}

static void
report_peer_table(mx_endpt_handle_t fd, uint32_t board_number, uint64_t nic_id,
		  int num_ports, int verbose, uint32_t max_peers, int all)
{
  mx_get_peer_format_t peer_format;
  mx_get_route_table_t get_route_table;
  char *r, *p, *pptr;
  int i, peer_count;
  uint32_t route_size;

  if (board_number >= MX_PEER_FLAG_SEEN_NB_BOARDS && !all) {
    fprintf(stderr, "Error, you need to use option -a when printing peer table for board numbers >= %d\n",
	    MX_PEER_FLAG_SEEN_NB_BOARDS);
    exit(1);
  }
    

  if (mx__get_peer_format(fd, &peer_format) != 0) {
    printf("MX_GET_PEER_FORMAT failed\n");
    exit(1);
  }
  route_size = board_number;
  if (mx__get_route_size(fd, &route_size) != 0) {
    printf("MX_GET_ROUTE_SIZE failed\n");
    exit(1);
  }

  p = malloc((size_t)(max_peers * peer_format.sizeof_peer_t));
  if (!p) {
    printf("Failed to allocate space for peer table\n");
    exit(1);
  }
  memset(p, 0, (size_t)(max_peers * peer_format.sizeof_peer_t));
  if (mx__get_peer_table(fd, p) != 0) {
    printf("Failed to get the peer table\n");
    exit(1);
  }
  
  r = malloc((size_t)((max_peers + 1) * route_size * num_ports));
  if (!r) {
    printf("Failed to allocate space for route table\n");
    exit(1);
  }
  memset(r, 0, (size_t)(max_peers * route_size * num_ports));
  get_route_table.board_number = board_number;
  get_route_table.routes = (uint64_t)(uintptr_t)r;

  if (mx__get_route_table(fd, &get_route_table) != 0) {
    printf("MX_GET_ROUTE_TABLE failed\n");
    exit(1);
  }

  pptr = p;
  peer_count = 0;
  for(i = 0; i < max_peers; i++) {
    mx_peer_t *peer = (mx_peer_t *)pptr;
    if (peer->mac_low32 != 0 && peer->mac_high16 != 0
	&& (all || (peer->flags & PEER_ACTIVE_MASK(board_number))))
      peer_count++;
    pptr += peer_format.sizeof_peer_t;
  }
  /*  printf("\t%d remote nodes seen\n", peer_count - instance_count);*/
  if (verbose) {
    for (i = 0; i < num_ports; i++) {
      printf("Routes for Port %d\n", i);
      parse_route_table(p, r, max_peers, route_size, peer_format, board_number, all);
      r += ((max_peers + 1) * route_size);
    }
  } else {
    if (namechars < 40)
      namechars = 40;
    parse_route_table_flat(p, r, max_peers, route_size, peer_format, 
			   num_ports, board_number, all);
  }
}
